"use client"

import { useState, useEffect } from "react"
import { Button } from "@/components/ui/button"
import { Card, CardContent } from "@/components/ui/card"
import { Badge } from "@/components/ui/badge"
import {
  Code2,
  Brain,
  Shield,
  Rocket,
  Database,
  Smartphone,
  Globe,
  Target,
  ChevronRight,
  Star,
  Zap,
  Lock,
  TrendingUp,
} from "lucide-react"

export default function HomePage() {
  const [isVisible, setIsVisible] = useState(false)

  useEffect(() => {
    setIsVisible(true)
  }, [])

  return (
    <div className="min-h-screen bg-gradient-to-br from-slate-900 via-purple-900 to-slate-900">
      {/* Navigation */}
      <nav className="fixed top-0 w-full z-50 bg-black/20 backdrop-blur-md border-b border-white/10">
        <div className="container mx-auto px-6 py-4">
          <div className="flex items-center justify-between">
            <div className="flex items-center space-x-2">
              <div className="w-8 h-8 bg-gradient-to-r from-cyan-400 to-purple-500 rounded-lg flex items-center justify-center">
                <Code2 className="w-5 h-5 text-white" />
              </div>
              <span className="text-xl font-bold text-white">TechForge</span>
            </div>
            <div className="hidden md:flex items-center space-x-8">
              <a href="#servicios" className="text-gray-300 hover:text-white transition-colors">
                Servicios
              </a>
              <a href="#especialidades" className="text-gray-300 hover:text-white transition-colors">
                Especialidades
              </a>
              <a href="#filosofia" className="text-gray-300 hover:text-white transition-colors">
                Filosofía
              </a>
              <a href="#contacto" className="text-gray-300 hover:text-white transition-colors">
                Contacto
              </a>
            </div>
            <Button className="bg-gradient-to-r from-cyan-500 to-purple-600 hover:from-cyan-600 hover:to-purple-700">
              Comenzar Proyecto
            </Button>
          </div>
        </div>
      </nav>

      {/* Hero Section */}
      <section className="relative min-h-screen flex items-center justify-center overflow-hidden">
        <div
          className="absolute inset-0 bg-cover bg-center opacity-30"
          style={{ backgroundImage: "url(/images/hero-ai-tech.png)" }}
        />
        <div className="absolute inset-0 bg-gradient-to-r from-black/50 to-transparent" />

        <div
          className={`relative z-10 text-center max-w-4xl mx-auto px-6 transition-all duration-1000 ${isVisible ? "opacity-100 translate-y-0" : "opacity-0 translate-y-10"}`}
        >
          <Badge className="mb-6 bg-gradient-to-r from-cyan-500/20 to-purple-500/20 text-cyan-300 border-cyan-500/30">
            <Zap className="w-4 h-4 mr-2" />
            Lanzamos startups cada semana
          </Badge>

          <h1 className="text-5xl md:text-7xl font-bold text-white mb-6 leading-tight">
            <span className="bg-gradient-to-r from-cyan-400 to-purple-500 bg-clip-text text-transparent">
              Inteligencia,
            </span>
            <br />
            <span className="text-white">Código y</span>
            <br />
            <span className="bg-gradient-to-r from-purple-400 to-pink-500 bg-clip-text text-transparent">
              Propósito
            </span>
          </h1>

          <p className="text-xl text-gray-300 mb-8 max-w-2xl mx-auto leading-relaxed">
            No programamos para ser empleados. Creamos soluciones sobresalientes que transforman industrias a través de
            IA, ciberseguridad y desarrollo de vanguardia.
          </p>

          <div className="flex flex-col sm:flex-row gap-4 justify-center">
            <Button
              size="lg"
              className="bg-gradient-to-r from-cyan-500 to-purple-600 hover:from-cyan-600 hover:to-purple-700 text-lg px-8 py-3"
            >
              Ver Nuestros Proyectos
              <ChevronRight className="w-5 h-5 ml-2" />
            </Button>
            <Button
              size="lg"
              variant="outline"
              className="border-white/20 text-white hover:bg-white/10 text-lg px-8 py-3 bg-transparent"
            >
              Conocer Más
            </Button>
          </div>
        </div>

        {/* Floating Elements */}
        <div className="absolute top-20 left-10 w-20 h-20 bg-gradient-to-r from-cyan-400/20 to-purple-500/20 rounded-full blur-xl animate-pulse" />
        <div className="absolute bottom-20 right-10 w-32 h-32 bg-gradient-to-r from-purple-400/20 to-pink-500/20 rounded-full blur-xl animate-pulse delay-1000" />
      </section>

      {/* Services Section */}
      <section id="servicios" className="py-20 relative">
        <div className="container mx-auto px-6">
          <div className="text-center mb-16">
            <h2 className="text-4xl font-bold text-white mb-4">
              Servicios de{" "}
              <span className="bg-gradient-to-r from-cyan-400 to-purple-500 bg-clip-text text-transparent">
                Vanguardia
              </span>
            </h2>
            <p className="text-gray-400 text-lg max-w-2xl mx-auto">
              Desarrollamos soluciones completas que van desde aplicaciones web hasta sistemas de IA avanzados
            </p>
          </div>

          <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-8">
            {[
              {
                icon: Globe,
                title: "Plataformas Web",
                description: "Aplicaciones web escalables y de alto rendimiento con tecnologías modernas",
                image: "/images/python-ai.png",
              },
              {
                icon: Smartphone,
                title: "Apps Móviles",
                description: "Aplicaciones móviles nativas y multiplataforma con UX excepcional",
                image: "/images/startup-innovation.png",
              },
              {
                icon: Brain,
                title: "Inteligencia Artificial",
                description: "Soluciones de IA personalizadas para automatización y análisis predictivo",
                image: "/images/python-ai.png",
              },
              {
                icon: Database,
                title: "Procesamiento de Datos",
                description: "Extracción, procesamiento y distribución de datos a gran escala",
                image: "/images/hero-ai-tech.png",
              },
              {
                icon: Shield,
                title: "Ciberseguridad",
                description: "Auditorías de seguridad y pentesting con certificación OSCP",
                image: "/images/cybersecurity.png",
              },
              {
                icon: Rocket,
                title: "Startups Propias",
                description: "Lanzamos productos innovadores cada semana con visión disruptiva",
                image: "/images/startup-innovation.png",
              },
            ].map((service, index) => (
              <Card
                key={index}
                className="bg-white/5 backdrop-blur-md border-white/10 hover:bg-white/10 transition-all duration-300 group overflow-hidden"
              >
                <div className="relative h-48 overflow-hidden">
                  <div
                    className="absolute inset-0 bg-cover bg-center group-hover:scale-110 transition-transform duration-500"
                    style={{ backgroundImage: `url(${service.image})` }}
                  />
                  <div className="absolute inset-0 bg-gradient-to-t from-black/80 to-transparent" />
                  <div className="absolute bottom-4 left-4">
                    <service.icon className="w-8 h-8 text-cyan-400" />
                  </div>
                </div>
                <CardContent className="p-6">
                  <h3 className="text-xl font-semibold text-white mb-3">{service.title}</h3>
                  <p className="text-gray-400 leading-relaxed">{service.description}</p>
                </CardContent>
              </Card>
            ))}
          </div>
        </div>
      </section>

      {/* Specialties Section */}
      <section id="especialidades" className="py-20 bg-black/20">
        <div className="container mx-auto px-6">
          <div className="text-center mb-16">
            <h2 className="text-4xl font-bold text-white mb-4">
              Nuestras{" "}
              <span className="bg-gradient-to-r from-purple-400 to-pink-500 bg-clip-text text-transparent">
                Especialidades
              </span>
            </h2>
            <p className="text-gray-400 text-lg max-w-2xl mx-auto">
              Dominamos las tecnologías más avanzadas para crear soluciones que marcan la diferencia
            </p>
          </div>

          <div className="grid md:grid-cols-2 gap-12 items-center">
            <div className="space-y-8">
              {[
                {
                  icon: Code2,
                  title: "Python & IA",
                  description: "Desarrollo avanzado en Python con frameworks de machine learning y deep learning",
                  skills: ["TensorFlow", "PyTorch", "Scikit-learn", "FastAPI"],
                },
                {
                  icon: Database,
                  title: "Big Data & Analytics",
                  description: "Procesamiento masivo de datos y generación de insights accionables",
                  skills: ["Apache Spark", "Pandas", "NumPy", "Data Pipelines"],
                },
                {
                  icon: Lock,
                  title: "Ciberseguridad OSCP",
                  description: "Auditorías de seguridad y pentesting con certificación profesional",
                  skills: ["Penetration Testing", "Vulnerability Assessment", "Security Audits", "OSCP Certified"],
                },
                {
                  icon: TrendingUp,
                  title: "Lead Generation",
                  description: "Sistemas automatizados de generación y calificación de leads",
                  skills: ["Web Scraping", "CRM Integration", "Marketing Automation", "Analytics"],
                },
              ].map((specialty, index) => (
                <div key={index} className="flex items-start space-x-4 group">
                  <div className="w-12 h-12 bg-gradient-to-r from-cyan-500 to-purple-600 rounded-lg flex items-center justify-center flex-shrink-0 group-hover:scale-110 transition-transform">
                    <specialty.icon className="w-6 h-6 text-white" />
                  </div>
                  <div>
                    <h3 className="text-xl font-semibold text-white mb-2">{specialty.title}</h3>
                    <p className="text-gray-400 mb-3">{specialty.description}</p>
                    <div className="flex flex-wrap gap-2">
                      {specialty.skills.map((skill, skillIndex) => (
                        <Badge
                          key={skillIndex}
                          variant="secondary"
                          className="bg-white/10 text-cyan-300 border-cyan-500/30"
                        >
                          {skill}
                        </Badge>
                      ))}
                    </div>
                  </div>
                </div>
              ))}
            </div>

            <div className="relative">
              <div
                className="rounded-2xl overflow-hidden shadow-2xl"
                style={{ backgroundImage: "url(/images/python-ai.png)" }}
              >
                <div className="aspect-square bg-cover bg-center relative">
                  <div className="absolute inset-0 bg-gradient-to-t from-black/60 to-transparent" />
                  <div className="absolute bottom-6 left-6 right-6">
                    <div className="bg-white/10 backdrop-blur-md rounded-lg p-4 border border-white/20">
                      <div className="flex items-center space-x-2 mb-2">
                        <Star className="w-5 h-5 text-yellow-400" />
                        <span className="text-white font-semibold">Certificación OSCP</span>
                      </div>
                      <p className="text-gray-300 text-sm">
                        Expertise certificado en ciberseguridad y pentesting profesional
                      </p>
                    </div>
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
      </section>

      {/* Philosophy Section */}
      <section id="filosofia" className="py-20">
        <div className="container mx-auto px-6">
          <div className="max-w-4xl mx-auto text-center">
            <h2 className="text-4xl font-bold text-white mb-8">
              Nuestra{" "}
              <span className="bg-gradient-to-r from-cyan-400 to-purple-500 bg-clip-text text-transparent">
                Filosofía
              </span>
            </h2>

            <div className="bg-white/5 backdrop-blur-md rounded-2xl p-8 border border-white/10 mb-12">
              <blockquote className="text-2xl text-gray-300 italic mb-6 leading-relaxed">
                "No programamos para ser empleados o por un pago por un servicio. Buscamos crear algo sobresaliente que
                transforme industrias y genere impacto real."
              </blockquote>
              <div className="w-16 h-1 bg-gradient-to-r from-cyan-400 to-purple-500 mx-auto" />
            </div>

            <div className="grid md:grid-cols-3 gap-8">
              {[
                {
                  icon: Brain,
                  title: "Inteligencia",
                  description: "Aplicamos conocimiento profundo y pensamiento estratégico en cada proyecto",
                },
                {
                  icon: Code2,
                  title: "Código",
                  description: "Escribimos código limpio, escalable y que resuelve problemas reales",
                },
                {
                  icon: Target,
                  title: "Propósito",
                  description: "Cada línea de código tiene un objetivo claro: crear valor e impacto",
                },
              ].map((value, index) => (
                <div key={index} className="text-center group">
                  <div className="w-16 h-16 bg-gradient-to-r from-cyan-500 to-purple-600 rounded-full flex items-center justify-center mx-auto mb-4 group-hover:scale-110 transition-transform">
                    <value.icon className="w-8 h-8 text-white" />
                  </div>
                  <h3 className="text-xl font-semibold text-white mb-3">{value.title}</h3>
                  <p className="text-gray-400">{value.description}</p>
                </div>
              ))}
            </div>
          </div>
        </div>
      </section>

      {/* CTA Section */}
      <section id="contacto" className="py-20 bg-gradient-to-r from-cyan-900/20 to-purple-900/20">
        <div className="container mx-auto px-6 text-center">
          <h2 className="text-4xl font-bold text-white mb-6">
            ¿Listo para crear algo{" "}
            <span className="bg-gradient-to-r from-cyan-400 to-purple-500 bg-clip-text text-transparent">
              sobresaliente
            </span>
            ?
          </h2>
          <p className="text-xl text-gray-300 mb-8 max-w-2xl mx-auto">
            Únete a nosotros en la creación de la próxima generación de soluciones tecnológicas
          </p>
          <div className="flex flex-col sm:flex-row gap-4 justify-center">
            <Button
              size="lg"
              className="bg-gradient-to-r from-cyan-500 to-purple-600 hover:from-cyan-600 hover:to-purple-700 text-lg px-8 py-3"
            >
              Iniciar Conversación
              <ChevronRight className="w-5 h-5 ml-2" />
            </Button>
            <Button
              size="lg"
              variant="outline"
              className="border-white/20 text-white hover:bg-white/10 text-lg px-8 py-3 bg-transparent"
            >
              Ver Portfolio
            </Button>
          </div>
        </div>
      </section>

      {/* Footer */}
      <footer className="py-12 bg-black/40 border-t border-white/10">
        <div className="container mx-auto px-6">
          <div className="flex flex-col md:flex-row items-center justify-between">
            <div className="flex items-center space-x-2 mb-4 md:mb-0">
              <div className="w-8 h-8 bg-gradient-to-r from-cyan-400 to-purple-500 rounded-lg flex items-center justify-center">
                <Code2 className="w-5 h-5 text-white" />
              </div>
              <span className="text-xl font-bold text-white">TechForge</span>
            </div>
            <p className="text-gray-400">© 2024 TechForge. Inteligencia, Código y Propósito.</p>
          </div>
        </div>
      </footer>
    </div>
  )
}
